import random, sys
import pygame.display
import map, entity, gui, combat
import cPickle as pickle


transfers = (
  ('village', 0),
  ('castle', 1),
  ('forest', 0),
  ('village', 1),
  ('river', 0),
  ('river', 1),
  ('cave', 0),
  ('grove', 1),
  ('grove', 0),
  ('forest', 1),
)


encounter_table = [
  # forest
  [combat.Goblin] * 4 + [combat.GoblinWarrior],
  [combat.Goblin, combat.GoblinWarrior] * 2 + [combat.GoblinElite],
  # river
  [combat.GoblinWarrior, combat.GoblinElite] * 2 + [combat.GoblinGhost],
  [combat.GoblinElite, combat.GoblinGhost, combat.ZombieWarrior],
  # grove
  [combat.ZombieWarrior] * 2 + [combat.WraithWarrior],
  # cave
  [combat.WraithWarrior] * 2 + [combat.GhostWarrior, combat.ZombieWarrior],
  [combat.GhostWarrior, combat.Shade],
]



npcs = (
  entity.King,
  entity.WeaponMerchant,
  entity.ArmorMerchant,
  entity.Ghost,
  entity.Healer,
  entity.PotionsMerchant,
  entity.Guard,
  entity.Peasant,
  entity.ForestHealer,
)


weapons = ['fist', 'dagger', 'short sword', 'long sword', 'holy sword']


armors = ['none', 'leather', 'chain', 'plate']


class Game:
  def __getstate__(self):
    d = self.__dict__.copy()
    d['map'] = self.map.name
    return d
  def __setstate__(self, state):
    self.__dict__.update(state)
    self.map = map.get_map(self.map)
  def __init__(self):
    self.hp = 10
    self.endurance = 1
    self.attack = 1
    self.defense = 1
    self.player = entity.Avatar(self, 0, 0)
    self.xp = 0
    self.gp = 5
    self.weapon = 0
    self.armor = 0
    self.state = 0
    self.potions = 0
    self.enter_map('castle', 0)
    self.ending = 0
    gui.message_screen('The Ghost of the Haunted Grove\n'
      'A game created in 48 hours.\n'
      'Copyright 2002 Rainer Deyke\n'
      'All rights reserved.\n\n')
    gui.message_screen('Prince Rupert is the only son of the king of a small, '
      'insignificant forest kingdom.  There is little of interested in the '
      'kingdom except the haunted grove.  And so, as Rupert grows from child '
      'to a youth hungry for adventure, it is only natural that he should '
      'seek out the haunted grove.')
  def enter_map(self, map_name, entry):
    self.map = map.get_map(map_name)
    entered = 0
    self.entities = [self.player]
    for iy in range(self.map.get_height()):
      for ix in range(self.map.get_width()):
        code = self.map.get_code(ix, iy)
        if code == entry + 0x10:
          assert not entered
          self.player.x = ix
          self.player.y = iy
          entered = 1
        elif code & 0x20:
          self.entities.append(npcs[code - 0x20](self, ix, iy))
        elif code & 0x40:
          self.entities.append(entity.TransferPoint(self, ix, iy,
            *transfers[code - 0x40]))
    assert entered
  def draw(self, target):
    base_x = min(max(0, self.player.x - 10), self.map.get_width() - 20)
    base_y = min(max(0, self.player.y - 7), self.map.get_height() - 15)
    for iy in range(15):
      self.map.draw(target, 0, iy * 16, base_x, base_y + iy, 20)
      for e in self.entities:
        if e.y == base_y + iy and e.x >= base_x and e.x < base_x + 20:
          e.draw(target, (e.x - base_x) * 16, (e.y - base_y) * 16)
    self.draw_status(target)
  def draw_status(self, target):
    s = 'hp: %3d ptns: %3d' % (self.hp, self.potions)
    target.fill((0xff, 0xff, 0xff), (0, 0, len(s) * 8 + 16, 24))
    target.fill((0, 0, 0), (4, 4, len(s) * 8 + 8, 16))
    gui.print_line(target, 8, 8, s)
  def run(self):
    while self.hp > 0 and not self.ending:
      self.draw(pygame.display.get_surface())
      pygame.display.update()
      for e in pygame.event.get():
        if e.type == pygame.KEYDOWN:
          if e.key == pygame.K_ESCAPE: return 1
          elif e.key == pygame.K_UP:
            self.player.move_up()
          elif e.key == pygame.K_DOWN:
            self.player.move_down()
          elif e.key == pygame.K_LEFT:
            self.player.move_left()
          elif e.key == pygame.K_RIGHT:
            self.player.move_right()
          elif e.key in (pygame.K_RETURN, pygame.K_KP_ENTER, pygame.K_SPACE):
            self.view_character()
        if e.type == pygame.QUIT: sys.exit()
    self.end_game()
  def end_game(self):
    if self.ending == 1: # killed king
      gui.message_screen('The battle rage wears off, leaving Rupert feeling '
        'hollow.  He looks at his father\'s corpse, and is flooded by mixed '
        'emotions.  Disgust... Guilt... Regret...  Triumph.  For the first '
        'time in his life, Rupert feels truly alone.  With icy determination, '
        'he takes his father\'s crown and places it on his own head.  He has '
        'done the right thing.  He has avenged his mother.  And he will face '
        'tomorrow with courage, as King Rupert.')
    elif self.ending == 2:
      gui.message_screen('As his mother\'s spirit falls screaming into the '
        'void and his battle rage wears off, Rupert is left feeling cold.  '
        'He has done his duty to his father and king.  He has betrayed his '
        'mother.  No blood stains his hands, yet he knows his soul is forever '
        'tarnished.  Is this then the price of duty?  With one last '
        'sigh he starts the long way back home.')
    elif self.hp <= 0:
      gui.message_screen('As Rupert\'s blood spills onto the ground, he gives '
        'one last sigh of regret.  Then awareness leaves him.  He died a '
        'warrior\'s death.')
  def set_weapon(self, n):
    self.attack -= self.weapon
    self.weapon = n
    self.attack += self.weapon
  def set_armor(self, n):
    self.defense -= self.armor
    self.armor = n
    self.defense += self.armor
  def view_character(self):
    screen = pygame.display.get_surface()
    selection = 0
    w = 32 * 8
    h = 12 * 8
    while 1:
      self.draw_status(screen)
      screen.fill((0xff, 0xff, 0xff),
        (160 - w / 2 - 8, 120 - h / 2 - 8, w + 16, h + 16))
      screen.fill((0, 0, 0),
        (160 - w / 2 - 4, 120 - h / 2 - 4, w + 8, h + 8))
      lines = ['Prince Rupert',
        'hp: %d/%d' % (self.hp, self.endurance * 10),
        'xp: %d' % self.xp,
        'gp: %d' % self.gp,
        'potions: %d' % self.potions,
        'weapon: %s' % weapons[self.weapon],
        'armor: %s' % armors[self.armor],
      ]
      prices = [4 * 2 ** (self.attack - self.weapon),
        4 * 2 ** (self.defense - self.armor), 4 * 2 ** self.endurance]
      options = ['attack: %d (+1 for %d xp)' % (self.attack, prices[0]),
        'defense: %d (+1 for %d xp)' % (self.defense, prices[1]),
        'endurance: %d (+1 for %d xp)' % (self.endurance, prices[2]),
        'drink potion']
      gui.print_lines(screen, 160 - w/2, 120 - h/2, lines)
      gui.print_lines(screen, 160 - w/2 + 16, 120 - h/2 + len(lines) * 8 + 8,
        options)
      gui.print_line(screen, 160 - w/2,
        120 - h/2 + (len(lines) + 1 + selection) * 8, '->')
      pygame.display.update()
      for e in pygame.event.get():
        if e.type == pygame.KEYDOWN:
          if e.key == pygame.K_UP:
            selection = (selection - 1) % len(options)
          elif e.key == pygame.K_DOWN:
            selection = (selection + 1) % len(options)
          elif e.key in (pygame.K_KP_ENTER, pygame.K_SPACE, pygame.K_RETURN):
            if selection == 3:
              if self.potions < 0:
                gui.message_box('You have no potions')
              elif self.hp >= self.endurance * 10:
                gui.message_box('You are not injured.')
              else:
                self.potions -= 1
                self.hp = min(self.hp + 50, self.endurance * 10)
            elif self.xp < prices[selection]:
              gui.message_box('Not enough xp.')
            else:
              self.xp -= prices[selection]
              if selection == 0:
                self.attack += 1
              elif selection == 1:
                self.defense += 1
              elif selection == 2:
                self.endurance += 1
          elif e.key == pygame.K_ESCAPE:
            return -1
        if e.type == pygame.QUIT: sys.exit()
  def update(self):
    if random.randrange(20) == 5:
      c = self.map.get_code(self.player.x, self.player.y)
      if c > 0 and c < 0x10:
        self.draw(pygame.display.get_surface())
        monster = random.choice(encounter_table[c - 1])
        monster(self).fight()
      



def play():
  game = Game()
  while 1:
    if not game.run(): break
    sel = gui.menu('Game menu', ['save', 'load', 'quit'])
    if sel == 0:
      pickle.dump(game, open('savegame', 'wb'), 1)
      gui.message_box('Game saved.')
    elif sel == 1:
      try:
        game = pickle.load(open('savegame', 'rb'))
      except:
        gui.message_box('Error loading game: %s' % sys.exc_info()[1])
      else:
        gui.message_box('Game loaded.')
    elif sel == 2:
      return
      
      


