import sys
import pygame.font, pygame.display
import map


font = map.get_tileset('font')


def print_line(target, x, y, line):
  for c in line:
    target.blit(font[ord(c) + 1], (x, y))
    x += 8


def print_lines(target, x, y, lines):
  for line in lines:
    print_line(target, x, y, line)
    y += 8


def split_text(s, length=32):
  lines = []
  for s in s.split('\n'):
    while len(s) > length:
      i = s.rfind(' ', 0, length) + 1
      if i == 0:
        i = length
      lines.append(s[:i])
      s = s[i:].strip()
    lines.append(s)
  return lines


def longest_line(lines): return max([len(line) for line in lines])


def message_box(text):
  lines = split_text(text)
  w = longest_line(lines) * 8
  h = len(lines) * 8
  screen = pygame.display.get_surface()
  screen.fill((0xff, 0xff, 0xff),
    (160 - w / 2 - 8, 120 - h / 2 - 8, w + 16, h + 16))
  screen.fill((0, 0, 0),
    (160 - w / 2 - 4, 120 - h / 2 - 4, w + 8, h + 8))
  print_lines(screen, 160 - w/2, 120 - h/2, lines)
  pygame.display.update()
  while 1:
    for e in pygame.event.get():
      if e.type == pygame.KEYDOWN: return
      if e.type == pygame.QUIT: sys.exit()


def menu(text, options, no_escape = 0):
  lines = split_text(text)
  w = max(longest_line(lines), longest_line(options) + 2) * 8
  h = (len(lines) + len(options) + 1) * 8
  screen = pygame.display.get_surface()
  selection = 0
  while 1:
    screen.fill((0xff, 0xff, 0xff),
      (160 - w / 2 - 8, 120 - h / 2 - 8, w + 16, h + 16))
    screen.fill((0, 0, 0),
      (160 - w / 2 - 4, 120 - h / 2 - 4, w + 8, h + 8))
    print_lines(screen, 160 - w/2, 120 - h/2, lines)
    print_lines(screen, 160 - w/2 + 16, 120 - h/2 + len(lines) * 8 + 8,
      options)
    print_line(screen, 160 - w/2,
      120 - h/2 + (len(lines) + 1 + selection) * 8, '->')
    pygame.display.update()
    for e in pygame.event.get():
      if e.type == pygame.KEYDOWN:
        if e.key == pygame.K_UP:
          selection = (selection - 1) % len(options)
        elif e.key == pygame.K_DOWN:
          selection = (selection + 1) % len(options)
        elif e.key in (pygame.K_KP_ENTER, pygame.K_SPACE, pygame.K_RETURN):
          return selection
        elif e.key == pygame.K_ESCAPE and not no_escape:
          return -1
      if e.type == pygame.QUIT: sys.exit()
  

class Console:
  def __init__(self, x, y, width, height):
    self.x = x
    self.y = y
    self.width = width
    self.height = height
    self.data = [''] * height
  def println(self, text):
    self.data = (self.data + split_text(text, self.width))[-self.height:]
  def draw(self, target):
    target.fill((0xff, 0xff, 0xff),
      (self.x - 8, self.y - 8, self.width * 8 + 16, self.height * 8 + 16))
    target.fill((0, 0, 0),
      (self.x - 4, self.y - 4, self.width * 8 + 8, self.height * 8 + 8))
    print_lines(target, self.x, self.y, self.data)
  def menu(self, options):
    self.data = (self.data + [''] * len(options))[-self.height:]
    selection = 0
    while 1:
      for i in range(len(options)):
        self.data[self.height - len(options) + i] = \
          ['  ', '->'][selection == i] + options[i]
      self.draw(pygame.display.get_surface())
      pygame.display.update()
      for e in pygame.event.get():
        if e.type == pygame.KEYDOWN:
          if e.key == pygame.K_UP:
            selection = (selection - 1) % len(options)
          elif e.key == pygame.K_DOWN:
            selection = (selection + 1) % len(options)
          elif e.key in (pygame.K_KP_ENTER, pygame.K_SPACE, pygame.K_RETURN):
            return selection
          elif e.key == pygame.K_ESCAPE:
            return -1
        if e.type == pygame.QUIT: sys.exit()
  def getch(self):
    self.draw(pygame.display.get_surface())
    pygame.display.update()
    while 1:
      for e in pygame.event.get():
        if e.type == pygame.KEYDOWN: return
        if e.type == pygame.QUIT: sys.exit()
  

def message_screen(text, mapname = 'introscreen'):
  lines = split_text(text, 320/8 - 2)
  screen = pygame.display.get_surface()
  map.get_map(mapname).draw(screen, 0, 0, 0, 0, 20, 15)
  print_lines(screen, 8, 8, lines)
  pygame.display.update()
  while 1:
    for e in pygame.event.get():
      if e.type == pygame.KEYDOWN: return
      if e.type == pygame.QUIT: sys.exit()
  
